//	TorusGames2DChess.c
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#include "TorusGames-Common.h"
#include "GeometryGamesUtilities-Common.h"
#include "GeometryGamesSound.h"
#include <math.h>


#define NUM_2D_BACKGROUND_TEXTURE_REPETITIONS_CHESS		4

//	When animating the computer's move,
//	how fast should the piece travel?
#define CHESS_PIECE_SPEED					2.0		//	squares per second

//	In the touch-based interface, when continuing a previously started move,
//	how close must the new touch be to the selected piece to count as a hit?
#define CHESS_MOVE_CONTINUATION_HIT_RADIUS	0.1

//	In the touch-based interface, the user taps a selected piece to put it down.
//	How much accidental movement should we allow in the definition of a tap?
#define CHESS_TAP_TOLERANCE					0.01


typedef struct
{
	signed int		itsDirections[8][2];
	unsigned int	itsNumDirections;	//	maximum of 8 valid directions
	bool			itsUnlimitedDistanceFlag;
} MoveRule;

typedef struct
{
	unsigned int	itsMovesConsidered,
					itsMovesTotal;
	double			*itsProgressMeter;
} ChessProgress;


static const MoveRule	gMoveRules[7] =
{
	//	unused
	{
		{{0,0},{0,0},{0,0},{0,0},{0,0},{0,0},{0,0},{0,0}},
		0,
		false
	},
	//	king
	{
		{{-1, 0},{+1, 0},{ 0,-1},{ 0,+1},{-1,-1},{-1,+1},{+1,+1},{+1,-1}},
		8,
		false
	},
	//	queen
	{
		{{-1, 0},{+1, 0},{ 0,-1},{ 0,+1},{-1,-1},{-1,+1},{+1,+1},{+1,-1}},
		8,
		true
	},
	//	bishop
	{
		{{-1,-1},{-1,+1},{+1,+1},{+1,-1},{0,0},{0,0},{0,0},{0,0}},
		4,
		true
	},
	//	knight
	{
		{{-2,-1},{-2,+1},{+2,-1},{+2,+1},{-1,-2},{+1,-2},{-1,+2},{+1,+2}},
		8,
		false
	},
	//	rook
	{
		{{-1, 0},{+1, 0},{ 0,-1},{ 0,+1},{0,0},{0,0},{0,0},{0,0}},
		4,
		true
	},
	//	pawn (capturing or not capturing)
	{
		{{-1, 0},{+1, 0},{ 0,-1},{ 0,+1},{-1,-1},{-1,+1},{+1,+1},{+1,-1}},
		8,
		false
	}
};

//	I have no idea how to accurately judge the relative values
//	of the pieces in torus or Klein bottle chess, so let's just
//	use the standard values and hope for the best.
//
//	gCheckmateBaseValue should exceed the combined value of all other pieces,
//	in effect making a checkmate more valuable than anything else.
//
//	gNoPlausibleMoveValue should be less than the value of any legal move.
//
//	Define gPieceValues as signed ints for easy negation.
//
static const signed int	gPieceValues[7] = {0, 0 /* king's value unused */, 9, 3, 3, 5, 1},
						gCheckmateBaseValue		=   128,	//	greater than any possible sum of piece values
						gNoPlausibleMoveValue	= -1024,	//	less than value of any legal move
						gInitialAlpha			= -2048,
						gInitialBeta			= +2048;


//	Public functions with private names
static void			ChessShutDown(ModelData *md);
static void			ChessReset(ModelData *md);
static bool			ChessDragBegin(ModelData *md, bool aRightClick);
static void			ChessDragObject(ModelData *md, double aHandLocalDeltaH, double aHandLocalDeltaV);
static void			ChessDragEnd(ModelData *md, double aDragDuration, bool aTouchSequenceWasCancelled);
static void			ChessSimulationUpdate(ModelData *md);

//	Private functions
static void			LetComputerMove1(ModelData *md);
static void			LetComputerMove2(ModelData *md);
static void			LetComputerMove3(ModelData *md);
static void			LetComputerMove4(ModelData *md);
static void			FindHitSquare(Placement2D *aPlacement, TopologyType aTopology, unsigned int *h, unsigned int *v, bool *aFlipFlag);
static bool			MoveIsPlausible(unsigned int aBoard[8][8], TopologyType aTopology, unsigned int h0, unsigned int v0, unsigned int h1, unsigned int v1, ChessMove *aMove);
static bool			KingWouldBeInCheck(unsigned int aBoard[8][8], TopologyType aTopology, unsigned int h0, unsigned int v0, unsigned int h1, unsigned int v1, unsigned int aThreatenedPlayer, ChessMove *aMove);
static bool			KingIsInCheck(unsigned int aBoard[8][8], TopologyType aTopology, unsigned int aThreatenedPlayer, ChessMove *aMove);
static void			TestForCheck(ModelData *md);
static bool			HasLegalMove(unsigned int aBoard[8][8], TopologyType aTopology, unsigned int aPlayer);
static void			ComputerSelectsMove(ModelData *md);
static signed int	ValueOfSituation(unsigned int aBoard[8][8], signed int aNetTotalCapturedPiecesValueSoFar,
						TopologyType aTopology, unsigned int aPlayer, unsigned int aRecursionHeight,
						signed int anAlpha, signed int aBeta,
						bool *anAbortFlag, ChessMove *aBestMove, ChessProgress *aProgressReport);
static void			TerminateThinkingThread(ModelData *md);
//static void		ChessComputerGoesFirst(ModelData *md);
static bool			AnimateComputerMove(ModelData *md);


void Chess2DSetUp(ModelData *md)
{
	//	Initialize function pointers.
	md->itsGameShutDown					= &ChessShutDown;
	md->itsGameReset					= &ChessReset;
	md->itsGameHumanVsComputerChanged	= NULL;
	md->itsGame2DHandMoved				= NULL;
	md->itsGame2DDragBegin				= &ChessDragBegin;
	md->itsGame2DDragObject				= &ChessDragObject;
	md->itsGame2DDragEnd				= &ChessDragEnd;
	md->itsGame3DDragBegin				= NULL;
	md->itsGame3DDragObject				= NULL;
	md->itsGame3DDragEnd				= NULL;
	md->itsGame3DGridSize				= NULL;
	md->itsGameCharacterInput			= NULL;
	md->itsGameSimulationUpdate			= &ChessSimulationUpdate;
	md->itsGameRefreshMessage			= NULL;
	
	//	Initialize flags.
	md->itsGameOf.Chess2D.itsUserMoveIsPending				= false;
	md->itsGameOf.Chess2D.itsComputerMoveIsPending			= false;
	md->itsGameOf.Chess2D.itsUserMoveBeganWithCurrentDrag	= false;
	md->itsGameOf.Chess2D.itsThreadThinkingFlag				= false;
	md->itsGameOf.Chess2D.itsThreadAbortFlag				= false;
	
	//	Set up the board.
	ChessReset(md);
}


static void ChessShutDown(ModelData *md)
{
	//	Terminate the thinking thread, if one is active.
	TerminateThinkingThread(md);
}


static void ChessReset(ModelData *md)
{
	unsigned int	h,
					v;

	//	Cancel any pending user move.
	md->itsGameOf.Chess2D.itsUserMoveIsPending = false;

	//	Abort any pending computer move.
	TerminateThinkingThread(md);
	SimulationEnd(md);
	md->itsGameOf.Chess2D.itsComputerMoveIsPending = false;
	
	//	Reset the pieces.

	for (h = 0; h < 8; h++)
		for (v = 0; v < 8; v++)
			md->itsGameOf.Chess2D.itsSquares[h][v] = CHESS_EMPTY_SQUARE;

	md->itsGameOf.Chess2D.itsSquares[1][1] = CHESS_PLAIN | CHESS_BLACK | CHESS_KNIGHT;
	md->itsGameOf.Chess2D.itsSquares[2][1] = CHESS_PLAIN | CHESS_BLACK | CHESS_PAWN  ;
	md->itsGameOf.Chess2D.itsSquares[3][1] = CHESS_PLAIN | CHESS_BLACK | CHESS_ROOK  ;
	md->itsGameOf.Chess2D.itsSquares[1][2] = CHESS_PLAIN | CHESS_BLACK | CHESS_BISHOP;
	md->itsGameOf.Chess2D.itsSquares[2][2] = CHESS_PLAIN | CHESS_BLACK | CHESS_KING  ;
	md->itsGameOf.Chess2D.itsSquares[3][2] = CHESS_PLAIN | CHESS_BLACK | CHESS_QUEEN ;
	md->itsGameOf.Chess2D.itsSquares[1][3] = CHESS_PLAIN | CHESS_BLACK | CHESS_ROOK  ;
	md->itsGameOf.Chess2D.itsSquares[2][3] = CHESS_PLAIN | CHESS_BLACK | CHESS_PAWN  ;
	md->itsGameOf.Chess2D.itsSquares[3][3] = CHESS_PLAIN | CHESS_BLACK | CHESS_KNIGHT;

	md->itsGameOf.Chess2D.itsSquares[5][5] = CHESS_PLAIN | CHESS_WHITE | CHESS_ROOK  ;
	md->itsGameOf.Chess2D.itsSquares[6][5] = CHESS_PLAIN | CHESS_WHITE | CHESS_PAWN  ;
	md->itsGameOf.Chess2D.itsSquares[7][5] = CHESS_PLAIN | CHESS_WHITE | CHESS_KNIGHT;
	md->itsGameOf.Chess2D.itsSquares[5][6] = CHESS_PLAIN | CHESS_WHITE | CHESS_QUEEN ;
	md->itsGameOf.Chess2D.itsSquares[6][6] = CHESS_PLAIN | CHESS_WHITE | CHESS_KING  ;
	md->itsGameOf.Chess2D.itsSquares[7][6] = CHESS_PLAIN | CHESS_WHITE | CHESS_BISHOP;
	md->itsGameOf.Chess2D.itsSquares[5][7] = CHESS_PLAIN | CHESS_WHITE | CHESS_KNIGHT;
	md->itsGameOf.Chess2D.itsSquares[6][7] = CHESS_PLAIN | CHESS_WHITE | CHESS_PAWN  ;
	md->itsGameOf.Chess2D.itsSquares[7][7] = CHESS_PLAIN | CHESS_WHITE | CHESS_ROOK  ;

	//	White goes first.
	md->itsGameOf.Chess2D.itsWhoseTurn = CHESS_WHITE;

	//	Noone threatens check.
	md->itsGameOf.Chess2D.itsCheckThreat.itsStartH = 0;
	md->itsGameOf.Chess2D.itsCheckThreat.itsStartV = 0;
	md->itsGameOf.Chess2D.itsCheckThreat.itsDeltaH = 0;
	md->itsGameOf.Chess2D.itsCheckThreat.itsDeltaV = 0;
	md->itsGameOf.Chess2D.itsCheckThreat.itsFinalH = 0;
	md->itsGameOf.Chess2D.itsCheckThreat.itsFinalV = 0;

	//	Game is live.
	md->itsGameIsOver = false;

#ifdef GAME_CONTENT_FOR_SCREENSHOT

	md->itsGameOf.Chess2D.itsSquares[5][5] = CHESS_EMPTY_SQUARE;
	md->itsGameOf.Chess2D.itsSquares[5][3] = CHESS_PLAIN | CHESS_WHITE | CHESS_ROOK  ;

	md->itsGameOf.Chess2D.itsSquares[3][3] = CHESS_EMPTY_SQUARE;
	md->itsGameOf.Chess2D.itsSquares[4][5] = CHESS_PLAIN | CHESS_BLACK | CHESS_KNIGHT;

	md->itsGameOf.Chess2D.itsCheckThreat.itsStartH = 4;
	md->itsGameOf.Chess2D.itsCheckThreat.itsStartV = 5;
	md->itsGameOf.Chess2D.itsCheckThreat.itsDeltaH = 2;
	md->itsGameOf.Chess2D.itsCheckThreat.itsDeltaV = 1;
	md->itsGameOf.Chess2D.itsCheckThreat.itsFinalH = 6;
	md->itsGameOf.Chess2D.itsCheckThreat.itsFinalV = 6;

#endif

#ifdef MAKE_GAME_CHOICE_ICONS

	md->itsGameOf.Chess2D.itsSquares[5][3] = CHESS_PLAIN | CHESS_BLACK | CHESS_ROOK  ;
	md->itsGameOf.Chess2D.itsSquares[6][2] = CHESS_PLAIN | CHESS_BLACK | CHESS_KNIGHT;

#endif

#if (SHAPE_OF_SPACE_CHESSBOARD == 1)	//	Figure 2.6

	for (h = 0; h < 8; h++)
		for (v = 0; v < 8; v++)
			md->itsGameOf.Chess2D.itsSquares[h][v] = CHESS_EMPTY_SQUARE;

	md->itsGameOf.Chess2D.itsSquares[0][7] = CHESS_PLAIN | CHESS_BLACK | CHESS_KNIGHT;
	md->itsGameOf.Chess2D.itsSquares[1][2] = CHESS_PLAIN | CHESS_BLACK | CHESS_KNIGHT;
	md->itsGameOf.Chess2D.itsSquares[2][6] = CHESS_PLAIN | CHESS_BLACK | CHESS_QUEEN ;
	md->itsGameOf.Chess2D.itsSquares[3][5] = CHESS_PLAIN | CHESS_BLACK | CHESS_ROOK  ;
	md->itsGameOf.Chess2D.itsSquares[5][7] = CHESS_PLAIN | CHESS_BLACK | CHESS_BISHOP;
	md->itsGameOf.Chess2D.itsSquares[6][3] = CHESS_PLAIN | CHESS_BLACK | CHESS_BISHOP;
	md->itsGameOf.Chess2D.itsSquares[6][7] = CHESS_PLAIN | CHESS_BLACK | CHESS_KING  ;
	md->itsGameOf.Chess2D.itsSquares[7][1] = CHESS_PLAIN | CHESS_WHITE | CHESS_KNIGHT;
	md->itsGameOf.Chess2D.itsSquares[7][7] = CHESS_PLAIN | CHESS_BLACK | CHESS_ROOK  ;

#endif
#if (SHAPE_OF_SPACE_CHESSBOARD == 2)	//	Figure 2.7

	for (h = 0; h < 8; h++)
		for (v = 0; v < 8; v++)
			md->itsGameOf.Chess2D.itsSquares[h][v] = CHESS_EMPTY_SQUARE;

	md->itsGameOf.Chess2D.itsSquares[0][5] = CHESS_PLAIN | CHESS_WHITE | CHESS_KNIGHT;
	md->itsGameOf.Chess2D.itsSquares[1][1] = CHESS_PLAIN | CHESS_WHITE | CHESS_QUEEN ;
	md->itsGameOf.Chess2D.itsSquares[1][3] = CHESS_PLAIN | CHESS_BLACK | CHESS_KING  ;
	md->itsGameOf.Chess2D.itsSquares[1][7] = CHESS_PLAIN | CHESS_BLACK | CHESS_ROOK  ;
	md->itsGameOf.Chess2D.itsSquares[6][4] = CHESS_PLAIN | CHESS_BLACK | CHESS_BISHOP;
	md->itsGameOf.Chess2D.itsSquares[6][6] = CHESS_PLAIN | CHESS_BLACK | CHESS_ROOK  ;
	md->itsGameOf.Chess2D.itsSquares[7][3] = CHESS_PLAIN | CHESS_BLACK | CHESS_KNIGHT;
	md->itsGameOf.Chess2D.itsSquares[7][7] = CHESS_PLAIN | CHESS_BLACK | CHESS_QUEEN  ;

#endif
#if (SHAPE_OF_SPACE_CHESSBOARD == 3)	//	Figure 2.8
	//	Keep the standard starting position.
#endif
#if (SHAPE_OF_SPACE_CHESSBOARD == 4)	//	Figure 4.8

	//	Draw an empty board.
	for (h = 0; h < 8; h++)
		for (v = 0; v < 8; v++)
			md->itsGameOf.Chess2D.itsSquares[h][v] = CHESS_EMPTY_SQUARE;

#endif
}


static bool ChessDragBegin(
	ModelData	*md,
	bool		aRightClick)
{
	unsigned int	h,
					v;
	bool			theNormalizationReverses;

	UNUSED_PARAMETER(aRightClick);

	//	The mouse-based and touch-based versions of torus chess handle drags differently.
#ifdef TORUS_GAMES_2D_TOUCH_INTERFACE

	//	Touch-based interface
	
	//	If a user move is already pending...
	if (md->itsGameOf.Chess2D.itsUserMoveIsPending)
	{
		//	Let ChessDragEnd() know that a move was pending
		//	even before this drag began.
		md->itsGameOf.Chess2D.itsUserMoveBeganWithCurrentDrag = false;

		//	If the user touches the selected piece...
		md->its2DDragIsReversed = (	md->its2DHandPlacement.itsFlip
								 != md->itsGameOf.Chess2D.itsMovePlacement.itsFlip);
		if (Shortest2DDistance(
					md->its2DHandPlacement.itsH,
					md->its2DHandPlacement.itsV,
					md->itsGameOf.Chess2D.itsMovePlacement.itsH,
					md->itsGameOf.Chess2D.itsMovePlacement.itsV,
					md->itsTopology,
					&md->its2DDragIsReversed)
				< CHESS_MOVE_CONTINUATION_HIT_RADIUS)
		{
			//	...then let the pending move continue.
			//	That is, start a new drag which will be part of the same pending move.
			md->itsGameOf.Chess2D.itsDragStartH = md->itsGameOf.Chess2D.itsMovePlacement.itsH;
			md->itsGameOf.Chess2D.itsDragStartV = md->itsGameOf.Chess2D.itsMovePlacement.itsV;
			md->itsGameOf.Chess2D.itsDragWasTap = true;
			return true;
		}
		else	//	But if the user touches elsewhere,
		{
			//	treat the touch as a scroll.
			//	The pre-existing move remains valid but suspended,
			//	awaiting future touches for its final completion.
			return false;
		}
	}
	
	//	No user move is pending, so let the code below start a new one.

#else
	//	Mouse-based interface
	//	Moves are never left in suspension, so no extra code is needed here.
#endif
	
	//	If we return false for any reason, then no move will be pending.
	md->itsGameOf.Chess2D.itsUserMoveIsPending				= false;
	md->itsGameOf.Chess2D.itsUserMoveBeganWithCurrentDrag	= false;

	//	If the game is already over, ignore all hits.
	if (md->itsGameIsOver)
		return false;

	//	What square got hit?
	FindHitSquare(&md->its2DHandPlacement, md->itsTopology, &h, &v, &theNormalizationReverses);

	//	If the square's empty, report no hit.
	if (md->itsGameOf.Chess2D.itsSquares[h][v] == CHESS_EMPTY_SQUARE)
		return false;

	//	Dismiss any previous threat graphics.
	md->itsGameOf.Chess2D.itsCheckThreat.itsDeltaH = 0;
	md->itsGameOf.Chess2D.itsCheckThreat.itsDeltaV = 0;

	//	If the player clicks on a piece of the wrong color...
	if ((md->itsGameOf.Chess2D.itsSquares[h][v] & CHESS_COLOR_MASK)
	 != (md->itsGameOf.Chess2D.itsWhoseTurn     & CHESS_COLOR_MASK))
	{
		//	Play a sound to make the error obvious.
		EnqueueSoundRequest(u"ChessWrongColor.mid");
		
		//	Do not start a move.

		//	Nevertheless accept the click as a drag so that we can highlight
		//	the correct pieces while the mouse remains down, to attract 
		//	the player's attention to them.
		return true;
	}

	md->itsGameOf.Chess2D.itsUserMoveIsPending				= true;
	md->itsGameOf.Chess2D.itsUserMoveBeganWithCurrentDrag	= true;

	md->itsGameOf.Chess2D.itsMoveStartH = h;
	md->itsGameOf.Chess2D.itsMoveStartV = v;

	md->itsGameOf.Chess2D.itsMovePlacement.itsH		= -0.5 + 0.125*h;
	md->itsGameOf.Chess2D.itsMovePlacement.itsV		= -0.5 + 0.125*v;
	md->itsGameOf.Chess2D.itsMovePlacement.itsFlip	= ((md->itsGameOf.Chess2D.itsSquares[h][v] & CHESS_PARITY_MASK) == CHESS_REFLECTED);
	md->itsGameOf.Chess2D.itsMovePlacement.itsAngle	= 0.0;
	md->itsGameOf.Chess2D.itsMovePlacement.itsSizeH	= 0.125;
	md->itsGameOf.Chess2D.itsMovePlacement.itsSizeV	= 0.125;

	md->itsGameOf.Chess2D.itsDragStartH	= md->itsGameOf.Chess2D.itsMovePlacement.itsH;
	md->itsGameOf.Chess2D.itsDragStartV	= md->itsGameOf.Chess2D.itsMovePlacement.itsV;
	md->itsGameOf.Chess2D.itsDragWasTap	= false;	//	A move's first drag never counts as a terminating tap.
													//	Instead, an initial tap serves to select the piece.

	md->its2DDragIsReversed = (	md->its2DHandPlacement.itsFlip
							  ^ md->itsGameOf.Chess2D.itsMovePlacement.itsFlip
							  ^ theNormalizationReverses);

	return true;
}


static void ChessDragObject(
	ModelData	*md,
	double		aHandLocalDeltaH,
	double		aHandLocalDeltaV)
{
	double	thePieceLocalDeltaH,
			thePieceLocalDeltaV,
			theHandAmbientDeltaH,
			theHandAmbientDeltaV,
			thePieceAmbientDeltaH,
			thePieceAmbientDeltaV;

	//	Move the hand.

	theHandAmbientDeltaH	= md->its2DHandPlacement.itsFlip
								? -aHandLocalDeltaH : aHandLocalDeltaH;
	theHandAmbientDeltaV	= aHandLocalDeltaV;

	md->its2DHandPlacement.itsH += theHandAmbientDeltaH;
	md->its2DHandPlacement.itsV += theHandAmbientDeltaV;
	Normalize2DPlacement(&md->its2DHandPlacement, md->itsTopology);

	//	Move the piece.

	if (md->itsGameOf.Chess2D.itsUserMoveIsPending)
	{
		thePieceLocalDeltaH		= md->its2DDragIsReversed
									? -aHandLocalDeltaH : aHandLocalDeltaH;
		thePieceLocalDeltaV		= aHandLocalDeltaV;

		thePieceAmbientDeltaH	= md->itsGameOf.Chess2D.itsMovePlacement.itsFlip
									? -thePieceLocalDeltaH : thePieceLocalDeltaH;
		thePieceAmbientDeltaV	= thePieceLocalDeltaV;

		md->itsGameOf.Chess2D.itsMovePlacement.itsH += thePieceAmbientDeltaH;
		md->itsGameOf.Chess2D.itsMovePlacement.itsV += thePieceAmbientDeltaV;
		Normalize2DPlacement(&md->itsGameOf.Chess2D.itsMovePlacement, md->itsTopology);

		//	If the piece has travelled a non-trivial distance
		//	from the start of the present drag, then the drag is not a tap.
		//	This is relevant only in the touch-based interface, not the mouse-based one.
		//	In the touch-based interface, a move may consist of several separate drags,
		//	terminated by a tap.
		if (Shortest2DDistance(
					md->itsGameOf.Chess2D.itsDragStartH,
					md->itsGameOf.Chess2D.itsDragStartV,
					md->itsGameOf.Chess2D.itsMovePlacement.itsH,
					md->itsGameOf.Chess2D.itsMovePlacement.itsV,
					md->itsTopology,
					NULL)
				> CHESS_TAP_TOLERANCE)
		{
			md->itsGameOf.Chess2D.itsDragWasTap = false;
		}
	}
}


static void ChessDragEnd(
	ModelData	*md,
	double		aDragDuration,	//	in seconds
	bool		aTouchSequenceWasCancelled)
{
	unsigned int	h,
					v;
	bool			theNormalizationReverses;

	UNUSED_PARAMETER(aDragDuration);

	//	If a touch sequence got cancelled (perhaps because a gesture was recognized)...
	if (aTouchSequenceWasCancelled)
	{
		//	If this touch sequence began the current pending move
		//	(most likely unintentionally)...
		if (md->itsGameOf.Chess2D.itsUserMoveBeganWithCurrentDrag)
		{
			//	...then cancel the current move.
			md->itsGameOf.Chess2D.itsUserMoveIsPending				= false;
			md->itsGameOf.Chess2D.itsUserMoveBeganWithCurrentDrag	= false;
		}
		else
		{
			//	The current move was pending even before this touch sequence began,
			//	so let it remain pending, awaiting subsequent drags.
		}
		
		//	Whether or not we just cancelled the move, we're done handling this drag.
		return;
	}

	//	Ignore drags that began on a piece of the wrong color.
	if ( ! md->itsGameOf.Chess2D.itsUserMoveIsPending )
		return;

	//	The mouse-based and touch-based versions of torus chess handle drags differently.
#ifdef TORUS_GAMES_2D_TOUCH_INTERFACE
	//	Touch-based interface
	
	//	If the drag was not a simple tap, then even though the drag is over,
	//	let the move remain pending, awaiting subsequent drags.
	if ( ! md->itsGameOf.Chess2D.itsDragWasTap )
		return;
	
	//	A simple tap ends both the drag and the move that the drag is a part of.
	//	Continue with the code below.
#else
	//	Mouse-based interface
	//	Each move consists of a single drag.
#endif
	
	//	The move is over.
	md->itsGameOf.Chess2D.itsUserMoveIsPending				= false;
	md->itsGameOf.Chess2D.itsUserMoveBeganWithCurrentDrag	= false;
	
	//	What square did we land on?
	FindHitSquare(&md->itsGameOf.Chess2D.itsMovePlacement, md->itsTopology, &h, &v, &theNormalizationReverses);

	//	If we're back to where we started,
	//	just update the flip and return.
	if (h == md->itsGameOf.Chess2D.itsMoveStartH
	 && v == md->itsGameOf.Chess2D.itsMoveStartV)
	{
		if (md->itsGameOf.Chess2D.itsMovePlacement.itsFlip
		  ^ theNormalizationReverses)
			md->itsGameOf.Chess2D.itsSquares[h][v] |= CHESS_PARITY_MASK;
		else
			md->itsGameOf.Chess2D.itsSquares[h][v] &= (~CHESS_PARITY_MASK);

		return;
	}

	//	A move is "plausible" if the piece can get from the old square
	//	to the new.  It's "legal" if, in addition, it does not put the
	//	player's king in check.

	//	If the move is implausible or illegal, ignore it.
	//	(Note:  No special action is required to ignore
	//	a move -- it happens automatically!)

	if ( ! MoveIsPlausible(	md->itsGameOf.Chess2D.itsSquares,
							md->itsTopology,
							md->itsGameOf.Chess2D.itsMoveStartH,
							md->itsGameOf.Chess2D.itsMoveStartV,
							h,
							v,
							NULL) )
		return;

	//	If KingWouldBeInCheck() reports a non-zero md->itsGameOf.Chess2D.itsCheckThreat,
	//	the standard drawing routine ChessDrawGL() will display the threat automatically.
	if (KingWouldBeInCheck(	md->itsGameOf.Chess2D.itsSquares,
							md->itsTopology,
							md->itsGameOf.Chess2D.itsMoveStartH,
							md->itsGameOf.Chess2D.itsMoveStartV,
							h,
							v,
							md->itsGameOf.Chess2D.itsWhoseTurn,
							&md->itsGameOf.Chess2D.itsCheckThreat))
	{
		EnqueueSoundRequest(u"ChessCheck.mid");
		return;
	}

	//	The move's both plausible and legal, so make it.
	md->itsGameOf.Chess2D.itsSquares[h][v] = md->itsGameOf.Chess2D.itsSquares[md->itsGameOf.Chess2D.itsMoveStartH][md->itsGameOf.Chess2D.itsMoveStartV];
	if (md->itsGameOf.Chess2D.itsMovePlacement.itsFlip ^ theNormalizationReverses)
		md->itsGameOf.Chess2D.itsSquares[h][v] |= CHESS_PARITY_MASK;
	else
		md->itsGameOf.Chess2D.itsSquares[h][v] &= (~CHESS_PARITY_MASK);
	md->itsGameOf.Chess2D.itsSquares[md->itsGameOf.Chess2D.itsMoveStartH][md->itsGameOf.Chess2D.itsMoveStartV] = CHESS_EMPTY_SQUARE;

	//	Now it's the other player's turn.
	md->itsGameOf.Chess2D.itsWhoseTurn ^= CHESS_COLOR_MASK;

	//	Test for check(mate).
	TestForCheck(md);

	//	If we're playing against the computer,
	//	let the computer make its move.
	if (md->itsHumanVsComputer
	 && ! md->itsGameIsOver )
	{
		LetComputerMove1(md);
	}
}


static void LetComputerMove1(ModelData *md)
{
	//	If we just put the computer's king in check...
	if (md->itsGameOf.Chess2D.itsCheckThreat.itsDeltaH != 0
	 || md->itsGameOf.Chess2D.itsCheckThreat.itsDeltaV != 0)
	{
		//	...give the "check sound" a moment to play before
		//	letting the computer make its move,
		SimulationBegin(md, Simulation2DChessWaitToMove);
	}
	else
	{
		//	...otherwise let the computer make its move immediately.
		LetComputerMove2(md);
	}
}


static void LetComputerMove2(ModelData *md)
{
	//	Let the computer do its thinking in a separate thread.
	//	After completing the move it will set itsThreadThinkingFlag = false.
	md->itsGameOf.Chess2D.itsThreadThinkingFlag	= true;
	md->itsGameOf.Chess2D.itsThreadAbortFlag	= false;
	StartNewThread(md, &ComputerSelectsMove);
	
	//	Let the main thread keep the user interface responsive
	//	while the separate thread selects a move.
	//	If the user cancels the game (for example if s/he
	//	resets Chess2D or selects a different game),
	//	set itsThreadAbortFlag to true and wait until
	//	the separate thread sets itsThreadThinkingFlag to false
	//	to signal completion (at which point we are guaranteed
	//	that it's done accessing shared data).
	SimulationBegin(md, Simulation2DChessChooseComputerMove);
}


static void LetComputerMove3(ModelData *md)
{
	//	The separate thread has terminated normally.
	//	Nevertheless, perform an unnecessary error check.
	if (md->itsGameOf.Chess2D.itsThreadAbortFlag)
		return;	//	should never occur

	//	Animate the move that the computer has chosen.
	md->itsGameOf.Chess2D.itsComputerMoveIsPending	= true;
	md->itsGameOf.Chess2D.itsMoveStartH				= md->itsGameOf.Chess2D.itsThreadBestMove.itsStartH;
	md->itsGameOf.Chess2D.itsMoveStartV				= md->itsGameOf.Chess2D.itsThreadBestMove.itsStartV;
	SimulationBegin(md, Simulation2DChessAnimateComputerMove);
}


static void LetComputerMove4(ModelData *md)
{
	ChessMove	theMove;

	//	The animation is complete.
	md->itsGameOf.Chess2D.itsComputerMoveIsPending = false;

	//	For convenience
	theMove = md->itsGameOf.Chess2D.itsThreadBestMove;

	//	Finalize the move.
	md->itsGameOf.Chess2D.itsSquares[theMove.itsFinalH][theMove.itsFinalV]
		= md->itsGameOf.Chess2D.itsSquares[theMove.itsStartH][theMove.itsStartV];
	if (md->itsTopology == Topology2DKlein
	 && (((theMove.itsStartV + theMove.itsDeltaV) - theMove.itsFinalV) & 0x00000008) )	//	moved an odd multiple of 8 ?
		md->itsGameOf.Chess2D.itsSquares[theMove.itsFinalH][theMove.itsFinalV] ^= CHESS_PARITY_MASK;
	md->itsGameOf.Chess2D.itsSquares[theMove.itsStartH][theMove.itsStartV] = CHESS_EMPTY_SQUARE;

	//	It's now the other player's turn.
	md->itsGameOf.Chess2D.itsWhoseTurn ^= CHESS_COLOR_MASK;
	
	//	Check?
	TestForCheck(md);
}


static void FindHitSquare(
	Placement2D		*aPlacement,	//	input
	TopologyType	aTopology,		//	input
	unsigned int	*h,				//	output
	unsigned int	*v,				//	output
	bool			*aFlipFlag)		//	output
{
	//	Guard against "impossible" errors.
	if (aPlacement->itsH < -0.5 || aPlacement->itsH > +0.5
	 || aPlacement->itsV < -0.5 || aPlacement->itsV > +0.5)
	{
		GeometryGamesErrorMessage(u"Invalid input in FindHitSquare() in TorusGames2DChess.c.", u"Internal Error");
		*h			= 0;
		*v			= 0;
		*aFlipFlag	= false;
		return;
	}

	*h = (unsigned int) floor(8*(aPlacement->itsH + 0.5) + 0.5);
	*v = (unsigned int) floor(8*(aPlacement->itsV + 0.5) + 0.5);
	*aFlipFlag = false;

	if (*v == 8)
	{
		if (aTopology == Topology2DKlein)
		{
			*h = 8 - *h;
			*aFlipFlag = true;
		}
		*v = 0;
	}
	if (*h == 8)
		*h = 0;
}


static bool MoveIsPlausible(
	unsigned int	aBoard[8][8],	//	like md->itsGameOf.Chess2D.itsSquares
	TopologyType	aTopology,
	unsigned int	h0,		//	start point
	unsigned int	v0,
	unsigned int	h1,		//	end point
	unsigned int	v1,
	ChessMove		*aMove)	//	output, may be NULL;  if non-NULL, search for shortest move, interpreting the "zero move" as "no move yet present"
{
	MoveRule		theMoveRule;
	bool			theResult;
	unsigned int	i,
					theCount;
	signed int		h,
					v,
					dh,
					dv,
					theDeltaH,
					theDeltaV;

	//	Just to be safe...
	if (aBoard[h0][v0] == CHESS_EMPTY_SQUARE)
		return false;

	//	Going nowhere does not count as a plausible move.
	if (h0 == h1 && v0 == v1)
		return false;

	//	A piece can't capture a piece of the same color.
	if (aBoard[h1][v1] != CHESS_EMPTY_SQUARE
	 && (aBoard[h0][v0] & CHESS_COLOR_MASK) == (aBoard[h1][v1] & CHESS_COLOR_MASK))
		return false;

	//	Select the appropriate MoveRule for the given piece.
	theMoveRule = gMoveRules[aBoard[h0][v0] & CHESS_PIECE_MASK];

	//	Set the default return value.
	theResult = false;

	//	Test all plausible destinations.
	for (i = 0; i < theMoveRule.itsNumDirections; i++)
	{
		h = h0;
		v = v0;
		dh = theMoveRule.itsDirections[i][0];
		dv = theMoveRule.itsDirections[i][1];
		theCount = 0;

		do
		{
			//	Take one step.
			h += dh;
			v += dv;
			theCount++;

			//	Stay within the canonical fundamental domain.
			if (v < 0)
			{
				v += 8;
				if (aTopology == Topology2DKlein)
				{
					h = 8 - h;
					dh = -dh;
				}
			}
			if (v > 7)
			{
				v -= 8;
				if (aTopology == Topology2DKlein)
				{
					h = 8 - h;
					dh = -dh;
				}
			}
			if (h < 0)
				h += 8;
			if (h > 7)
				h -= 8;

			//	Did we reach the desired ending square?
			if ((unsigned int)h == h1
			 && (unsigned int)v == v1)
			{
				//	For a typical piece we've found our mark,
				//	but a pawn may capture iff it moves on a diagonal.
				if
				(
					(aBoard[h0][v0] & CHESS_PIECE_MASK) == CHESS_PAWN
				 &&
						(aBoard[h1][v1] != CHESS_EMPTY_SQUARE)
					 != (dh != 0 && dv != 0)
				)
					return false;

				//	We've found our mark.
				theResult = true;

				//	Report the move if requested.
				if (aMove != NULL)
				{
					theDeltaH = (signed int)theCount * theMoveRule.itsDirections[i][0];
					theDeltaV = (signed int)theCount * theMoveRule.itsDirections[i][1];

					//	If...
					if
					(
						//	...no move is yet present,
						(aMove->itsDeltaH == 0 && aMove->itsDeltaV == 0)
					 ||
						//	...or the existing move is longer,
						(	ABS(aMove->itsDeltaH) + ABS(aMove->itsDeltaV)
						  >    ABS(theDeltaH)     +    ABS(theDeltaV)    )
					)
					{
						//	...then record this move.
						aMove->itsStartH = h0;
						aMove->itsStartV = v0;
						aMove->itsDeltaH = theDeltaH;
						aMove->itsDeltaV = theDeltaV;
						aMove->itsFinalH = h1;
						aMove->itsFinalV = v1;
					}
				}

				//	If we don't care about the particular move,
				//	we can return immediately.  Otherwise keep going,
				//	in case there's a shorter way to get to the same place.
				if (aMove == NULL)
					return true;
			}

		} while (theMoveRule.itsUnlimitedDistanceFlag
			  && aBoard[h][v] == CHESS_EMPTY_SQUARE);
	}

	return theResult;
}


static bool KingWouldBeInCheck(
	unsigned int	aBoard[8][8],	//	like md->itsGameOf.Chess2D.itsSquares
	TopologyType	aTopology,
	unsigned int	h0,	//	start point
	unsigned int	v0,
	unsigned int	h1,	//	end point
	unsigned int	v1,
	unsigned int	aThreatenedPlayer,	//	CHESS_WHITE or CHESS_BLACK
	ChessMove		*aMove)				//	output;  may be NULL
{
	unsigned int	theRememberedPiece;
	bool			theKingWouldBeInCheck;

	//	Make the proposed move.
	theRememberedPiece	= aBoard[h1][v1];
	aBoard[h1][v1]		= aBoard[h0][v0];
	aBoard[h0][v0]		= CHESS_EMPTY_SQUARE;

	//	Test whether it puts aThreatenedPlayer's king in check.
	theKingWouldBeInCheck = KingIsInCheck(aBoard, aTopology, aThreatenedPlayer, aMove);

	//	Un-make the proposed move.
	aBoard[h0][v0]		= aBoard[h1][v1];
	aBoard[h1][v1]		= theRememberedPiece;

	return theKingWouldBeInCheck;
}


static bool KingIsInCheck(
	unsigned int	aBoard[8][8],		//	like md->itsGameOf.Chess2D.itsSquares
	TopologyType	aTopology,
	unsigned int	aThreatenedPlayer,	//	CHESS_WHITE or CHESS_BLACK
	ChessMove		*aMove)				//	output;  may be NULL
{
	unsigned int	h,
					v,
					h1,
					v1,
					theOpponent;

	//	For safety, and to keep the compiler happy.
	h1 = 0;
	v1 = 0;

	//	Locate the king.
	for (h = 0; h < 8; h++)
		for (v = 0; v < 8; v++)
			if ((aBoard[h][v] & (CHESS_COLOR_MASK | CHESS_PIECE_MASK)) == (aThreatenedPlayer | CHESS_KING))
			{
				h1 = h;
				v1 = v;
			}

	//	Identify the opponent.
	theOpponent = aThreatenedPlayer ^ CHESS_COLOR_MASK;

	//	Clear aMove, if present.
	if (aMove != NULL)
	{
		aMove->itsStartH = 0;
		aMove->itsStartV = 0;
		aMove->itsDeltaH = 0;
		aMove->itsDeltaV = 0;
		aMove->itsFinalH = 0;
		aMove->itsFinalV = 0;
	}

	//	Check whether any of the opponent's pieces could capture the king.
	//
	//	Note:  This is a grossly inefficient algorithm for testing for check.
	//	A far more efficient algorithm would be to start with the king and ask
	//
	//	1.	Would a pawn-move   (starting from the king) land on an enemy pawn?
	//	2.	Would a knight-move (starting from the king) land on an enemy knight?
	//	3.	Would a rook-move   (starting from the king) land on an enemy rook or queen?
	//	4.	Would a bishop-move (starting from the king) land on an enemy bishop or queen?
	//
	//	At the moment, however, this routine never gets called
	//	within any time-critical code, so I'm leaving the following algorithm
	//	in place because the code is simpler and shorter.
	//	(But if ever the more efficient algorithm is needed,
	//	it won't be ridiculously complicated either.)

	for (h = 0; h < 8; h++)
		for (v = 0; v < 8; v++)
			if (aBoard[h][v] != CHESS_EMPTY_SQUARE
			 && (aBoard[h][v] & CHESS_COLOR_MASK) == theOpponent
			 && MoveIsPlausible(aBoard, aTopology, h, v, h1, v1, aMove))
			{
				//	If we're not recording the move, we're done.
				if (aMove == NULL)
					return true;

				//	Otherwise keep searching in hopes of finding a shorter move.
			}

	if (aMove == NULL)
		return false;
	else
		return aMove->itsDeltaH != 0 || aMove->itsDeltaV != 0;
}


static void TestForCheck(ModelData *md)
{
	if (KingIsInCheck(md->itsGameOf.Chess2D.itsSquares, md->itsTopology, md->itsGameOf.Chess2D.itsWhoseTurn, &md->itsGameOf.Chess2D.itsCheckThreat))
	{
		if (HasLegalMove(md->itsGameOf.Chess2D.itsSquares, md->itsTopology, md->itsGameOf.Chess2D.itsWhoseTurn))
		{
			EnqueueSoundRequest(u"ChessCheck.mid");
		}
		else
		{
			md->itsGameIsOver						= true;
			md->itsGameOf.Chess2D.itsCheckmateFlag	= true;
			EnqueueSoundRequest(u"ChessCheckmate.mid");
		}
	}
	else
	{
		if ( ! HasLegalMove(md->itsGameOf.Chess2D.itsSquares, md->itsTopology, md->itsGameOf.Chess2D.itsWhoseTurn) )
		{
			md->itsGameIsOver						= true;
			md->itsGameOf.Chess2D.itsCheckmateFlag	= false;
			EnqueueSoundRequest(u"ChessStalemate.mid");
		}
	}
}


static bool HasLegalMove(
	unsigned int	aBoard[8][8],	//	like md->itsGameOf.Chess2D.itsSquares
	TopologyType	aTopology,
	unsigned int	aPlayer)		//	CHESS_WHITE or CHESS_BLACK
{
	//	This algorithm is rather inefficient, but that's OK given that
	//	we use it only once per move to detect checkmate/stalemate.
	//	If we ever needed to use it within a time-critical loop, 
	//	we'd need to design a more efficient algorithm.
	//	For present purposes, let's stick with simplicity.

	unsigned int	h0,
					v0,
					h1,
					v1;

	for (h0 = 0; h0 < 8; h0++)
		for (v0 = 0; v0 < 8; v0++)
			if (aBoard[h0][v0] != CHESS_EMPTY_SQUARE
			 && (aBoard[h0][v0] & CHESS_COLOR_MASK) == aPlayer)
				for (h1 = 0; h1 < 8; h1++)
					for (v1 = 0; v1 < 8; v1++)
						if (MoveIsPlausible(aBoard, aTopology, h0, v0, h1, v1, NULL)
						 && ! KingWouldBeInCheck(aBoard, aTopology, h0, v0, h1, v1, aPlayer, NULL) )
							return true;

	return false;
}


static void ComputerSelectsMove(ModelData *md)
{
	bool			*theThinkingFlag,
					*theAbortFlag;
	ChessMove		*theBestMove;
	bool			*theProgressMeterFlag;
	double			*theProgressMeter;
	TopologyType	theTopology;
	unsigned int	theWhoseTurn,			//	CHESS_WHITE or CHESS_BLACK
					theDifficultyLevel;		//	0…3
	ChessProgress	theDryRunProgressReport,
					theProgressReport;
	double			theDryRunProgressMeter;
	unsigned int	theBoard[8][8],
					h,
					v,
					theRecursionDepth;
	signed int		theValue;

	//	This function will run in its own separate thread,
	//	while the main thread keeps the UI responsive.
	//	If a torus cursor is present, the main thread will
	//	display it as a watch (i.e. a progress meter).

	//	Identify the shared variables
	//	that we'll use to communicate with the main thread.
	theThinkingFlag			= &md->itsGameOf.Chess2D.itsThreadThinkingFlag;
	theAbortFlag			= &md->itsGameOf.Chess2D.itsThreadAbortFlag;
	theBestMove				= &md->itsGameOf.Chess2D.itsThreadBestMove;
	theProgressMeterFlag	= &md->itsProgressMeterIsActive;
	theProgressMeter		= &md->itsProgressMeter;

	//	Make a copy of the chess board
	//	along with other read-only variables.
	theTopology			= md->itsTopology;
	theWhoseTurn		= md->itsGameOf.Chess2D.itsWhoseTurn;
	theDifficultyLevel	= md->itsDifficultyLevel;
	for (h = 0; h < 8; h++)
		for (v = 0; v < 8; v++)
			theBoard[h][v] = md->itsGameOf.Chess2D.itsSquares[h][v];

	//	Hereafter we ignore md.
	//	Our only interaction with its contents is
	//	via the shared variables defined previously.

	//	Activate the progress meter.
	*theProgressMeter		= 0.0;
	*theProgressMeterFlag	= true;

	//	Measure progress as a fraction of the total number
	//	of moves to be considered.  Make a dry run at recursion depth 0
	//	to count the total number of initial moves to be considered.
	theDryRunProgressReport.itsMovesConsidered	= 0;
	theDryRunProgressReport.itsMovesTotal		= 1;	//	value is irrelevant just so we avoid division by zero
	theDryRunProgressReport.itsProgressMeter	= &theDryRunProgressMeter;
	(void) ValueOfSituation(
			theBoard,
			0,
			theTopology,
			theWhoseTurn,
			0,
			gInitialAlpha,
			gInitialBeta,
			NULL,
			NULL,
			&theDryRunProgressReport);
	theProgressReport.itsMovesConsidered	= 0;
	theProgressReport.itsMovesTotal			= theDryRunProgressReport.itsMovesConsidered;
	theProgressReport.itsProgressMeter		= theProgressMeter;

	//	Assign a recursion depth to each difficulty level.
	//
	//		Note:  The number of moves to be considered
	//		will be theRecursionDepth + 1.
	//
	switch (theDifficultyLevel)
	{
		//	Keep the Easy level truly easy,
		//	to give beginners and children a chance to win.
		//
		//		Note:  theRecursionDepth must be at least 1,
		//		meaning that the algorithm will consider its own move
		//		plus the opponent's response, to ensure that
		//		the algorithm never makes a move that "lets its king
		//		get captured", in other words, to ensure that
		//		it never makes a move that leaves its king in check.
		//
		case 0: theRecursionDepth = 1; break; // 2 moves deep: black, white
		case 1: theRecursionDepth = 3; break; // 4 moves deep: black, white, black, white
		case 2: theRecursionDepth = 4; break; // 5 moves deep: black, white, black, white, black
		case 3: theRecursionDepth = 5; break; // 6 moves deep: black, white, black, white, black, white

		default: theRecursionDepth = 1; break;	//	should never occur
	}

	//	Redundant but safe
	if (theRecursionDepth < 1)	//	theRecursionDepth = 0 could leave king exposed to check
		theRecursionDepth = 1;
	if (theRecursionDepth > 5)	//	performance is acceptable when theRecursionDepth ≤ 5
		theRecursionDepth = 5;

	//	Let ValueOfSituation() select the best move.
	theValue = ValueOfSituation(
		theBoard,
		0,
		theTopology,
		theWhoseTurn,
		theRecursionDepth,
		gInitialAlpha,
		gInitialBeta,
		theAbortFlag,
		theBestMove,
		&theProgressReport);

	//	In theory we could report theValue to the user,
	//	which might give some idea of how well the game
	//	is going from the computer's point of view,
	//	but for now let's just ignore it.
	(void) theValue;

	//	De-activate the progress meter.
	*theProgressMeterFlag	= false;
	*theProgressMeter		= 0.0;

	//	We're done.  This thread is about to terminate.
	//	As our last act, let the main thread know it may proceed.
	*theThinkingFlag = false;
}


//	ValueOfSituation() starts with aBoard and lets aPlayer and
//	his/her opponent play a total of (aRecursionHeight + 1) moves.
//	For example, if aRecursionHeight is 2, the moves will be taken by
//
//		aPlayer, the opponent, aPlayer
//
//	while if aRecursionHeight is 3, the moves will be taken by
//
//		aPlayer, the opponent, aPlayer, the opponent.
//
//	ValueOfSituation() typically returns the "net captured piece value",
//	defined as
//
//			(the total value of all pieces that aPlayer has captured)
//		minus
//			(the total value of all pieces that the opponent has captured)
//
//	after those (aRecursionHeight + 1) moves have been completed,
//	assuming optimal play by both aPlayer and the opponent.
//	Exception:  In cases where a king "gets captured" along the way,
//	ValueOfSituation() immediately returns
//
//			gCheckmateBaseValue + aRecursionHeight
//
//	The reason for adding in aRecursionHeight is to favor early wins
//	(as a courtesy to the opponent) and late losses (to give the opponent
//	a chance to go wrong).
//
//	Note #1
//	The return value alternates sign as it gets passed back up the tree,
//	because it gets considered by aPlayer and the opponent in alternation.
//
//	Note #2
//	The net captured piece value is measured from the beginning of the recursion
//	(the root of the tree), instead of from the current node,
//	to faciliate alpha-beta pruning (see Note #3 below).
//
//	Note #3
//	The algorithm uses alpha-beta pruning.  To understand αβ pruning,
//	it's useful to think in terms of the search tree's "fully determined nodes"
//	and "partially determined nodes".
//
//		A "fully determined node" is one for which the value
//		of the resulting board is known, assuming optimal play
//		by both aPlayer and the opponent for all remaining moves.
//		A player who deviates from the optimal sequence of moves
//		might do worse than the determined value, but cannot do better.
//
//		A "partially determined node" is one for which the optimal result
//		has been determined for some (or none) of its available moves,
//		but not yet for all of them.
//
//	The insight of αβ pruning is that even a partially determined node
//	has some information available to it:  All of its ancestors (higher up
//	in the search tree) are also partially determined nodes, and typically
//	they have already examined some of their other children.  In other words,
//	those ancestor nodes have already examined some of the other moves available
//	to the player at that point in the game.  To encapsulate that information,
//	we may define
//
//		α to be the best result that the current player (the player whose possible
//			moves we're evaluating at the current node in the search tree)
//			could have obtained by choosing a different move at a higher node
//			in the search tree (that is, by making a different move
//			earlier in the game), and
//
//		β to be the worst result that the current opponent could
//			limit the current player to, by choosing a different move
//			at a higher node in the search tree.
//
//	Those ancestor nodes are all only partially determined, so α gives only
//	a lower bound on what aPlayer can achieve, and β gives only an upper bound
//	on what the opponent can limit aPlayer to.  Nevertheless, it's useful information:
//	if, while evaluating the moves available at the current node, we find
//	a move whose value exceeds β, we'll know that our opponent will never
//	let us reach this node -- the opponent will instead choose a different move
//	at an ancestor node, to limit us to a result of value β or less.
//	So we can stop processing the current node immediately, without wasting
//	any more time on it.  That's the heart of αβ pruning.
//
//	Note that we made no use of α there.  α comes into play as we work our way
//	down the search tree.  A positive result for us is a negative result for our opponent,
//	that is, for our opponent, α' and β' are given by
//
//		α' = -β
//		β' = -α
//
//	In practice, αβ pruning is wonderfully effective.
//	A search that took six minutes on my iPhone 5s without αβ pruning,
//	takes only 1 second with it.  αβ pruning could be even more effective
//	in combination with some sort of method that would let us examine
//	the most promising moves first, but for an app like the Torus Games,
//	I think keeping the code simple is more important than trying to write
//	the strongest possible Torus Chess algorithm.
//
//	Note #4
//	The β cutoff is never reached at the root level of the search tree,
//	so we're sure to evaluate all possible first moves.  If there's more than one
//	optimal first move, the following code chooses one of them at random.
//	Thus the human player won't always get the same response to a given sequence
//	of moves, which adds bit of variety to the game.
//
static signed int ValueOfSituation(	//	Returns value of board after (aRecursionHeight + 1) additional moves,
									//	 assuming optimal play by both players (see comment immediately above).
	unsigned int	aBoard[8][8],		//	input; like md->itsGameOf.Chess2D.itsSquares
	signed int		aNetTotalCapturedPiecesValueSoFar,	//	input; net value of captured pieces before any further moves are taken
	TopologyType	aTopology,			//	input
	unsigned int	aPlayer,			//	input; CHESS_BLACK or CHESS_WHITE
	unsigned int	aRecursionHeight,	//	input
	signed int		anAlpha,			//	input; see detailed explanation above
	signed int		aBeta,				//	input; see detailed explanation above
	bool			*anAbortFlag,		//	input; may be NULL
	ChessMove		*aBestMove,			//	output; is NULL except at root level of search tree
	ChessProgress	*aProgressReport)	//	input and output; is NULL except at root level of search tree
{
	signed int		theBestValue;
	unsigned int	theBestValueMultiplicity,	//	how many different optimal moves share theBestValue;
												//		used only at root level of search tree,
												//		ignored at all other nodes
					h0,
					v0;
	MoveRule		theMoveRule;
	unsigned int	i;
	signed int		h1,
					v1,
					dh,
					dv;
	unsigned int	theCount,
					theRememberedPiece;
	signed int		theNetTotalCapturedPiecesValueFinal,
					theCaptureValue;

	//	Has the user cancelled?
	if (anAbortFlag != NULL && *anAbortFlag)
	{
		if (aBestMove != NULL)	//	true only at root level of search tree
			*aBestMove = (ChessMove) {0, 0, 0, 0, 0, 0};

		return 0;
	}

	//	Performance note:
	//		Rather than testing each move to see whether it leaves
	//		the player's king in check, let's simply ignore any possible
	//		exposure to check and let the recursion catch it at the next
	//		deeper level, where it will manifest itself as an opportunity
	//		for the opponent to capture the player's king.
	//		For this approach to work, we must ensure that
	//		the initial recursion depth is at least 1,
	//		so that if we move our king into check on our first move,
	//		our opponent will have a chance to capture it and report the win.
	//		And indeed ComputerSelectsMove() ensures that.

	theBestValue				= gNoPlausibleMoveValue;
	theBestValueMultiplicity	= 0;	//	used only at root level of search tree, ignored at all other nodes

	for (h0 = 0; h0 < 8; h0++)
	{
		for (v0 = 0; v0 < 8; v0++)
		{
			if (aBoard[h0][v0] != CHESS_EMPTY_SQUARE
			 && (aBoard[h0][v0] & CHESS_COLOR_MASK) == aPlayer)
			{
				//	Select the appropriate MoveRule for the given piece.
				theMoveRule = gMoveRules[aBoard[h0][v0] & CHESS_PIECE_MASK];

				//	Test all plausible destinations.
				for (i = 0; i < theMoveRule.itsNumDirections; i++)
				{
					h1 = h0;
					v1 = v0;
					dh = theMoveRule.itsDirections[i][0];
					dv = theMoveRule.itsDirections[i][1];
					theCount = 0;

					do
					{
						//	Take one step.
						h1 += dh;
						v1 += dv;
						theCount++;

						//	Stay within the canonical fundamental domain.

						//		In a Klein bottle chess game, whenever a piece
						//		wraps around in the vertical direction,
						//		it flips in the horizontal direction.
						if (aTopology == Topology2DKlein)
						{
							if ( (unsigned int)v1 & 0xFFFFFFF8 )	//	equivalent to (v1 < 0 || v1 > 7)
							{
								h1 = 8 - h1;
								dh = -dh;
							}
						}
						
						//		Now that we've taken care of a possible flip,
						//		we wrap the coordinates to the fundamental domain.
						//
						//			Note:  The line
						//
						//				h1 = (signed int)( (unsigned int)h1 & 0x00000007 );
						//
						//			is logically equivalent to code like
						//
						//				if (h1 < 0)
						//					h1 += 8;
						//				if (h1 > 7)
						//					h1 -= 8;
						//				h1 = (unsigned int) h1;
						//
						//			and similarly for v1.
						//
						h1 = (signed int)( (unsigned int)h1 & 0x00000007 );	//	typecasts to make intended meaning clear
						v1 = (signed int)( (unsigned int)v1 & 0x00000007 );	//	typecasts to make intended meaning clear

						//	Ignore illegal pawn moves.
						if ((aBoard[h0][v0] & CHESS_PIECE_MASK) == CHESS_PAWN
						 && (aBoard[h1][v1] != CHESS_EMPTY_SQUARE) != (dh != 0 && dv != 0))
							continue;

						//	Ignore illegal captures.
						if (aBoard[h1][v1] != CHESS_EMPTY_SQUARE
						 && (aBoard[h1][v1] & CHESS_COLOR_MASK) == aPlayer)
							continue;

						//	Note the value of a captured piece, if any.
						theCaptureValue = gPieceValues[aBoard[h1][v1] & CHESS_PIECE_MASK];

						//	Evaluate the results of this move.
						if ((aBoard[h1][v1] & CHESS_PIECE_MASK) == CHESS_KING)
						{
							//	We've "captured" the opponent's king, so the game is over.
							//
							//	In general, whenever we can win, we'd rather do so
							//	sooner (fewer moves) than later (more moves),
							//	as a courtesy to our opponent.
							//	Conversely, when we're facing a loss, we'd rather lose
							//	later (more moves) than sooner (fewer moves), to give
							//	our opponent a chance to go wrong.  Both these goals
							//	are achieved if, rather than reporting
							//
							//		aNetTotalCapturedPiecesValueSoFar + theCaptureValue
							//
							//	we instead report
							//
							//		gCheckmateBaseValue + aRecursionHeight
							//
							//	gCheckmateBaseValue is already larger than any possible
							//	total captured pieces value, and adding in aRecursionHeight
							//	makes an earlier win (higher recursion height) more attractive
							//	than a later one (lower recursion height).
							//
							//	And, of course, a slightly larger positive value for one player
							//	is a slightly more negative value for the other, so an early win
							//	by our opponent would be seen as less desirable than a later win
							//	by our opponent.
							//
							theNetTotalCapturedPiecesValueFinal	= gCheckmateBaseValue + aRecursionHeight;
						}
						else
						if (aRecursionHeight == 0)
						{
							//	stop the recursion
							theNetTotalCapturedPiecesValueFinal = aNetTotalCapturedPiecesValueSoFar + theCaptureValue;
						}
						else
						{
							//	Let the recursion run for aRecursionHeight more steps.
							//	Let theNetTotalCapturedPiecesValueFinal be the net value of captured pieces,
							//	as seen from our perspective.

							//	Temporarily make the proposed move.
							theRememberedPiece	= aBoard[h1][v1];
							aBoard[h1][v1]		= aBoard[h0][v0];
							aBoard[h0][v0]		= CHESS_EMPTY_SQUARE;

							//	The situation's value to aPlayer is
							//	the negative of its value to the opponent.
							theNetTotalCapturedPiecesValueFinal =
								- ValueOfSituation(
									aBoard,
									-(aNetTotalCapturedPiecesValueSoFar + theCaptureValue),
									aTopology,
									aPlayer^CHESS_COLOR_MASK,
									aRecursionHeight - 1,
									-aBeta, -anAlpha,
									anAbortFlag, NULL, NULL);

							//	Un-make the proposed move.
							aBoard[h0][v0]		= aBoard[h1][v1];
							aBoard[h1][v1]		= theRememberedPiece;
						}

						//	Is this move better than all the alternatives examined so far?
						if (theNetTotalCapturedPiecesValueFinal > theBestValue)
						{
							theBestValue				= theNetTotalCapturedPiecesValueFinal;
							theBestValueMultiplicity	= 0;
							
							//	If we're at the root level of the search tree,
							//	we'll also execute the code immediately below,
							//	bracketed by
							//
							//		if (aBestMove != NULL)
							//		{
							//			if (theNetTotalCapturedPiecesValueFinal == theBestValue)
							//			{
							//				...
							//			}
							//		}
							//
							//	Among other things, that code will increment
							//	theBestValueMultiplicity to 1.
						}
						
						//	If we're at the root level of the search tree,
						//	the caller will want to know a move that realizes theBestValue.
						if (aBestMove != NULL)	//	true only at root level of search tree
						{
							if (theNetTotalCapturedPiecesValueFinal == theBestValue)
							{
								theBestValueMultiplicity++;

								//	Choose randomly among the equally good moves.
								if ( RandomUnsignedInteger() % theBestValueMultiplicity == 0 )
								{
									aBestMove->itsStartH = h0;
									aBestMove->itsStartV = v0;
									aBestMove->itsDeltaH = theCount * theMoveRule.itsDirections[i][0];
									aBestMove->itsDeltaV = theCount * theMoveRule.itsDirections[i][1];
									aBestMove->itsFinalH = h1;
									aBestMove->itsFinalV = v1;
								}
							}
						}
						
						//	If we're at the root level of the search tree,
						//	let the UI know about our progress.
						if (aProgressReport != NULL)	//	true only at root level of search tree
						{
							if (aProgressReport->itsMovesTotal != 0)	//	unnecessary but safe
							{
								aProgressReport->itsMovesConsidered++;
								*(aProgressReport->itsProgressMeter)
									= (double) aProgressReport->itsMovesConsidered
									/ (double) aProgressReport->itsMovesTotal;
							}
						}

						//	Have we found a move that's better than our known lower bound α ?
						if (theBestValue > anAlpha)
						{
							anAlpha = theBestValue;

							if (aBestMove != NULL)	//	true only at root level of search tree
							{
								//	We want to fully evaluate all maximally good moves
								//	the root level of the search tree, so that we may
								//	choose one of them randomly (see code above).
								//	The hope is that the human player will enjoy
								//	having the game make different responses
								//	to the same set of initial moves.
								//	In order to keep analyzing moves of equal value
								//	at the root level, it suffices to artifically
								//	decrement α by one.
								anAlpha--;
							}
						
							//	If α ≥ β, then there's no point in evaluating
							//	aPlayer's remaining possible moves at this node,
							//	because our opponent can make a different move,
							//	at a higher node in the search tree, to obtain
							//	a result of β.
							if (anAlpha >= aBeta)
								goto ReturnTheBestValueFoundSoFar;
						}

					} while (	theMoveRule.itsUnlimitedDistanceFlag
							&& aBoard[h1][v1] == CHESS_EMPTY_SQUARE);
				}
			}
		}
	}
	
	//	We can reach this point for one of two reasons:
	//	either the nested loops have run their full course,
	//	or we reach a point where α ≥ β and immediately broke out
	//	of all the nested loops.
ReturnTheBestValueFoundSoFar:

	if (theBestValue != gNoPlausibleMoveValue)
	{
		return theBestValue;
	}
	else
	{
		//	Given that the above code is perfectly willing
		//	to let our king move into check (and get captured
		//	at the next level in the recursion), it's almost
		//	unimaginable that we wouldn't have at least one move
		//	available.
		return 0;	//	should never occur
	}
}


static void TerminateThinkingThread(ModelData *md)
{
	//	If a thinking thread exists...
	if (md->itsGameOf.Chess2D.itsThreadThinkingFlag)
	{
		//	...politely ask it to stop as quickly as possible and then...
		md->itsGameOf.Chess2D.itsThreadAbortFlag = true;
		
		//	...wait for it to do so.  The reason we wait here
		//	for the thread to stop is that we want to rigorously guarantee
		//	that the main thread can't start a new thinking thread
		//	while the old thinking thread is still using 
		//	the shared communication variables.
		while (md->itsGameOf.Chess2D.itsThreadThinkingFlag)
			SleepBriefly();
	}
}


//	Torus Games once had a Computer Moves First menu item
//	to ask the computer to play the white pieces in Chess2D.
//	However, the feature was rarely used,
//	so I removed it to simplify the menus.
//	If I ever want that feature back, 
//	here's the function that implements it:
//
//	static void ChessComputerGoesFirst(ModelData *md)
//	{
//		if (md->itsHumanVsComputer)
//		{
//			ChessReset(md);
//			LetComputerMove1(md);
//		}
//	}


static void ChessSimulationUpdate(ModelData *md)
{
	switch (md->itsSimulationStatus)
	{
		case Simulation2DChessWaitToMove:
			//	Give the "check sound" a second to play before
			//	letting the computer make its move.
			if (md->itsSimulationElapsedTime >= 1.0)
			{
				SimulationEnd(md);
				LetComputerMove2(md);
			}
			break;
		
		case Simulation2DChessChooseComputerMove:
		
			//	While the computer chooses its move in a separate thread,
			//	the main thread is keeping the user interface responsive.
			//	This is good but we don't want to hog too many cycles.
			//	So let's let the main thread take a little nap.
			SleepBriefly();

			//	When the computer has finished thinking...
			if ( ! md->itsGameOf.Chess2D.itsThreadThinkingFlag )
			{
				//	...let it make the chosen move.
				SimulationEnd(md);
				LetComputerMove3(md);
			}

			break;
		
		case Simulation2DChessAnimateComputerMove:
			//	Draw the moving piece.
			//	When the motion is complete...
			if (AnimateComputerMove(md))
			{
				//	...finalize the move.
				SimulationEnd(md);
				LetComputerMove4(md);
			}
			break;

		default:
			break;
	}
}


static bool AnimateComputerMove(ModelData *md)
{
	ChessMove	theMove;
	double		theTotalDistance,
				theTotalTime,
				dh,
				dv;
	
	//	For convenience
	theMove = md->itsGameOf.Chess2D.itsThreadBestMove;

	//	How far will the piece move?
	theTotalDistance = sqrt(theMove.itsDeltaH * theMove.itsDeltaH
						  + theMove.itsDeltaV * theMove.itsDeltaV);
	if (theTotalDistance == 0.0)
	{
		GeometryGamesErrorMessage(u"theTotalDistance is zero in AnimateComputerMove().", u"Internal Error");
		return true;	//	Give up on the animation.
	};
	
	//	How long would we like it to take?
	theTotalTime = theTotalDistance / CHESS_PIECE_SPEED;
	
	//	Has the full time elapsed?
	if (md->itsSimulationElapsedTime >= theTotalTime)
		return true;	//	The animation is complete.

	//	Compute (dh, dv)...
	
	//	...as a unit vector
	dh = (theMove.itsDeltaH / theTotalDistance);
	dv = (theMove.itsDeltaV / theTotalDistance);
	
	//	...as a velocity vector
	dh *= CHESS_PIECE_SPEED;
	dv *= CHESS_PIECE_SPEED;
	
	//	...as a distance vector
	dh *= md->itsSimulationElapsedTime;
	dv *= md->itsSimulationElapsedTime;

	//	Set up the placement.
	md->itsGameOf.Chess2D.itsMovePlacement.itsH		= -0.5 + 0.125*(theMove.itsStartH + dh);
	md->itsGameOf.Chess2D.itsMovePlacement.itsV		= -0.5 + 0.125*(theMove.itsStartV + dv);
	md->itsGameOf.Chess2D.itsMovePlacement.itsFlip	= ((md->itsGameOf.Chess2D.itsSquares[theMove.itsStartH][theMove.itsStartV] & CHESS_PARITY_MASK) == CHESS_REFLECTED);
	md->itsGameOf.Chess2D.itsMovePlacement.itsAngle	= 0.0;
	md->itsGameOf.Chess2D.itsMovePlacement.itsSizeH	= 0.125;
	md->itsGameOf.Chess2D.itsMovePlacement.itsSizeV	= 0.125;

	//	Normalize to the fundamental domain.
	Normalize2DPlacement(&md->itsGameOf.Chess2D.itsMovePlacement, md->itsTopology);
	
	//	Because md->itsSimulationStatus != SimulationNone,
	//	the board will get redrawn.
	//	Because md->itsGameOf.Chess2D.itsComputerMoveIsPending == true,
	//	the drag will get displayed properly.
	//	No further action is required here!

	return false;	//	Keep going with the animation.
}


unsigned int GetNum2DChessBackgroundTextureRepetitions(void)
{
	return NUM_2D_BACKGROUND_TEXTURE_REPETITIONS_CHESS;
}

void Get2DChessKleinAxisColors(
	float	someKleinAxisColors[2][4])	//	output;  premultiplied alpha
{
	someKleinAxisColors[0][0] = 1.00;
	someKleinAxisColors[0][1] = 0.00;
	someKleinAxisColors[0][2] = 0.00;
	someKleinAxisColors[0][3] = 1.00;

	someKleinAxisColors[1][0] = 0.00;
	someKleinAxisColors[1][1] = 0.00;
	someKleinAxisColors[1][2] = 1.00;
	someKleinAxisColors[1][3] = 1.00;
}


unsigned int GetNum2DChessSprites(void)
{
	return 8*8	//	cells
		  + 4	//	piece-in-motion, arrow, target, mate
		  + 5;	//	progress indicator dial (base, sweep A, sweep B, rim and arrow)
}

void Get2DChessSpritePlacements(
	ModelData		*md,				//	input
	unsigned int	aNumSprites,		//	input
	Placement2D		*aPlacementBuffer)	//	output;  big enough to hold aNumSprites Placement2D's
{
	unsigned int	i,
					h,
					v;
	
	static const unsigned int	n				= 8;	//	chess board is 8 squares by 8 squares
	static const double			theCellSize		= 1.0 / (double) n;

	GEOMETRY_GAMES_ASSERT(
		md->itsGame == Game2DChess,
		"Game2DChess must be active");

	GEOMETRY_GAMES_ASSERT(
		aNumSprites == GetNum2DChessSprites(),
		"Internal error:  wrong buffer size");

	//	cells
	//		note (h,v) indexing -- not (row,col)
	for (h = 0; h < n; h++)
	{
		for (v = 0; v < n; v++)
		{
			i = n*h + v;
			aPlacementBuffer[i].itsH		= -0.5 + h * theCellSize;
			aPlacementBuffer[i].itsV		= -0.5 + v * theCellSize;
			aPlacementBuffer[i].itsFlip		= (md->itsGameOf.Chess2D.itsSquares[h][v] & CHESS_PARITY_MASK) == CHESS_REFLECTED;
			aPlacementBuffer[i].itsAngle	= 0.0;
			aPlacementBuffer[i].itsSizeH	= theCellSize;
			aPlacementBuffer[i].itsSizeV	= theCellSize;
		}
	}
	
	//	piece-in-motion, if any
	if (md->itsGameOf.Chess2D.itsUserMoveIsPending
	 || md->itsGameOf.Chess2D.itsComputerMoveIsPending)
	{
		aPlacementBuffer[8*8 + 0] = md->itsGameOf.Chess2D.itsMovePlacement;
	}
	else
	{
		aPlacementBuffer[8*8 + 0] = gUnusedPlacement;
	}
	
	//	check
	if (md->itsGameOf.Chess2D.itsCheckThreat.itsDeltaH != 0
	 || md->itsGameOf.Chess2D.itsCheckThreat.itsDeltaV != 0)
	{
		//	arrow
		aPlacementBuffer[8*8 + 1].itsH		= -0.5 + md->itsGameOf.Chess2D.itsCheckThreat.itsStartH*0.125;
		aPlacementBuffer[8*8 + 1].itsV		= -0.5 + md->itsGameOf.Chess2D.itsCheckThreat.itsStartV*0.125;
		aPlacementBuffer[8*8 + 1].itsFlip	= false;
		aPlacementBuffer[8*8 + 1].itsAngle	= atan2(md->itsGameOf.Chess2D.itsCheckThreat.itsDeltaV, md->itsGameOf.Chess2D.itsCheckThreat.itsDeltaH);
		aPlacementBuffer[8*8 + 1].itsSizeH	= 1.5 * 0.125;
		aPlacementBuffer[8*8 + 1].itsSizeV	= 1.5 * 0.125;
		
		//	target
		aPlacementBuffer[8*8 + 2].itsH		= -0.5 + md->itsGameOf.Chess2D.itsCheckThreat.itsFinalH*0.125;
		aPlacementBuffer[8*8 + 2].itsV		= -0.5 + md->itsGameOf.Chess2D.itsCheckThreat.itsFinalV*0.125;
		aPlacementBuffer[8*8 + 2].itsFlip	= false;
		aPlacementBuffer[8*8 + 2].itsAngle	= 0.0;
		aPlacementBuffer[8*8 + 2].itsSizeH	= 1.5 * 0.125;
		aPlacementBuffer[8*8 + 2].itsSizeV	= 1.5 * 0.125;
	}
	else
	{
		aPlacementBuffer[8*8 + 1] = gUnusedPlacement;
		aPlacementBuffer[8*8 + 2] = gUnusedPlacement;
	}
	
	//	mate (checkmate or stalemate)
	if (md->itsGameIsOver)
	{
		//	In the case of a checkmate the king is known to be at
		//	(itsGameOf.Chess2D.itsCheckThreat.itsFinalH, itsGameOf.Chess2D.itsCheckThreat.itsFinalV)
		//	but in the case of a stalemate we don't know where it is,
		//	so we must find it manually.
		aPlacementBuffer[8*8 + 3].itsH = 0.0;	//	initialize for safety only
		aPlacementBuffer[8*8 + 3].itsV = 0.0;
		for (h = 0; h < 8; h++)
		{
			for (v = 0; v < 8; v++)
			{
				if ((md->itsGameOf.Chess2D.itsSquares[h][v] & (CHESS_COLOR_MASK | CHESS_PIECE_MASK))
				 == (md->itsGameOf.Chess2D.itsWhoseTurn | CHESS_KING))
				{
					aPlacementBuffer[8*8 + 3].itsH = -0.5 + h*0.125;
					aPlacementBuffer[8*8 + 3].itsV = -0.5 + v*0.125;
				}
			}
		}
		aPlacementBuffer[8*8 + 3].itsFlip	= false;
		aPlacementBuffer[8*8 + 3].itsAngle	= 0.0;
		aPlacementBuffer[8*8 + 3].itsSizeH	= 1.5 * 0.125;
		aPlacementBuffer[8*8 + 3].itsSizeV	= 1.5 * 0.125;
	}
	else
	{
		aPlacementBuffer[8*8 + 3] = gUnusedPlacement;
	}
	
	//	progress indicator dial
	if (md->itsSimulationStatus == Simulation2DChessChooseComputerMove
	 && md->itsProgressMeterIsActive)
	{
		//
		//	To draw the progress indicator at the center of the display,
		//	subtract off the fundamental domain's translation and possible flip.
		//
		
		//	base
		aPlacementBuffer[8*8 + 4 + 0].itsH		= md->itsOffset.itsFlip ? +md->itsOffset.itsH : -md->itsOffset.itsH;
		aPlacementBuffer[8*8 + 4 + 0].itsV		= -md->itsOffset.itsV;
		aPlacementBuffer[8*8 + 4 + 0].itsFlip	= md->itsOffset.itsFlip;
		aPlacementBuffer[8*8 + 4 + 0].itsAngle	= 0.0;
		aPlacementBuffer[8*8 + 4 + 0].itsSizeH	= 0.25;
		aPlacementBuffer[8*8 + 4 + 0].itsSizeV	= 0.25;
		
		//	sweep A and sweep B
		aPlacementBuffer[8*8 + 4 + 1] = aPlacementBuffer[8*8 + 4 + 0];
		aPlacementBuffer[8*8 + 4 + 2] = aPlacementBuffer[8*8 + 4 + 0];
		if (md->itsProgressMeter <= 0.5)
		{
			aPlacementBuffer[8*8 + 4 + 1].itsAngle	= -TWOPI * md->itsProgressMeter;
			aPlacementBuffer[8*8 + 4 + 2].itsAngle	= -TWOPI * 0.0;
		}
		else
		{
			aPlacementBuffer[8*8 + 4 + 1].itsAngle	= -TWOPI * 0.5;
			aPlacementBuffer[8*8 + 4 + 2].itsAngle	= -TWOPI * md->itsProgressMeter;
		}
		
		//	rim
		aPlacementBuffer[8*8 + 4 + 3] = aPlacementBuffer[8*8 + 4 + 0];
		
		//	arrow
		aPlacementBuffer[8*8 + 4 + 4] = aPlacementBuffer[8*8 + 4 + 0];
		aPlacementBuffer[8*8 + 4 + 4].itsAngle	= -TWOPI * md->itsProgressMeter;
		aPlacementBuffer[8*8 + 4 + 4].itsSizeH	*= 0.75;
		aPlacementBuffer[8*8 + 4 + 4].itsSizeV	*= 0.75;
	}
	else
	{
		aPlacementBuffer[8*8 + 4 + 0] = gUnusedPlacement;
		aPlacementBuffer[8*8 + 4 + 1] = gUnusedPlacement;
		aPlacementBuffer[8*8 + 4 + 2] = gUnusedPlacement;
		aPlacementBuffer[8*8 + 4 + 3] = gUnusedPlacement;
		aPlacementBuffer[8*8 + 4 + 4] = gUnusedPlacement;
	}
}
